//
//  RMGeometry.h
//  RMFoundation
//
//  Created by Keith Duncan on 19/03/2010.
//  Copyright 2010 Realmac Software. All rights reserved.
//

#import <TargetConditionals.h>

#import <Foundation/Foundation.h>
#import <QuartzCore/QuartzCore.h>

#import "RMFoundation/RMFoundation.h"

/*!
	\brief
	Returns the mid point of the rect.
 */
NS_INLINE CGPoint AFRectGetCenterPoint(CGRect rect) {
	return CGPointMake(CGRectGetMidX(rect), CGRectGetMidY(rect));
}

/*!
	\brief
	Center a size around a given point. This provides the size with an origin.
 */
NS_INLINE CGRect AFSizeCenteredAroundPoint(CGSize size, CGPoint point) {
	return CGRectMake(floor(point.x - (size.width / 2.)), floor(point.y - (size.height / 2.)), size.width, size.height);
}

/*!
	\brief
	Center a size around the center point of a rect.
 */
NS_INLINE CGRect AFRectCenteredSize(CGRect frame, CGSize size) {
	return AFSizeCenteredAroundPoint(size, AFRectGetCenterPoint(frame));
}

/*!
	\brief
	Center a square size in the middle of a rectangle.
 */
NS_INLINE CGRect AFRectCenteredSquare(CGRect frame, CGFloat squareSize) {
	return AFSizeCenteredAroundPoint(CGSizeMake(squareSize, squareSize), AFRectGetCenterPoint(frame));
}

/*!
	\brief
	Center a rectangle around a center point, uses the size from the `frame` parameter and calculates a new origin.
 */
NS_INLINE CGRect AFRectCenteredAroundPoint(CGRect frame, CGPoint point) {
	return AFSizeCenteredAroundPoint(frame.size, point);
}

/*!
	\brief
	Centers a rect inside another, can be used to center a rectangle in the co-ordinate space of it's parent.
 */
NS_INLINE CGRect AFRectCenteredRect(CGRect frame, CGRect bounds) {
	return AFSizeCenteredAroundPoint(bounds.size, AFRectGetCenterPoint(frame));
}

NS_INLINE CGSize RMSizeScale(CGSize size, CGFloat scale) {
	return CGSizeMake(floor(size.width * scale), floor(size.height * scale));
}

NS_INLINE CGRect RMRectRound(CGRect rect) {
	return CGRectMake(round(rect.origin.x), round(rect.origin.y), ceil(rect.size.width), ceil(rect.size.height));
}

/*!
	\brief
	This functions divides the given rect into |count| pieces and stores them in |buffer|.
	Buffer must be large enough to store (|count| * sizeof(CGRect)).
 */
extern void AFRectDivideEqually(CGRect rect, CGRectEdge edge, NSUInteger count, CGRect *buffer);

/*!
	\brief
	This scales in aspect ratio but no bigger than the source size.
 */
extern NSString *const kRMGravityResizeAspectMax;
/*!
	\param gravity
	One of the gravity constants defined in CALayer.h.
 */
extern CGRect RMSizeScaledToRect(CGRect frame, CGSize size, NSString *gravity);

/*!
	\param point
	The point to test if it is on the edges of the CGRect
	
	\param rect
	The CGRect to test against.
 */
extern BOOL RMPointIsOnRectEdge(CGPoint point, CGRect rect);

/*!
	\param thePoint 
	The CGPoint to move until it lies within boundingRect.
	
	\param boundingRect
	The CGRect to move theRect to lie within.
 */
extern CGPoint RMPointMoveToLieWithinRect(CGPoint const thePoint, CGRect const boundingRect);

/*!
	\param theRect 
	The CGRect to move until it lies within boundingRect.
	
	\param boundingRect
	The CGRect to move theRect to lie within.
 */
extern CGRect RMRectMoveToLieWithinRect(CGRect const theRect, CGRect const boundingRect);

/*!
	\brief
	Moves the coordinate the minimum amount, length can be zero, the amount moved is returned byref
	
	\param offsetRef
	Can be NULL
 */
extern CGFloat RMCoordinateMoveToLieWithSizeInRange(CGFloat coordinate, CGFloat length, CGFloat minmax[2], CGFloat *offsetRef);

typedef RM_ENUM(NSUInteger, RMRectCorner) {
	RMRectCornerBottomLeft  = 0,
	RMRectCornerBottomRight = 1,
	RMRectCornerTopRight    = 2,
	RMRectCornerTopLeft     = 3,
};
/*!
	\param theRect 
	The existing CGRect
	
	\param endPoint
	The new position of the control point
	
	\param cornerIndex
	The CGPoint of where the dragging ended.
	
	\param outerBounds
	A CGRect representing the outer bounds of the Rect being dragged.
	The rect will be constrained to always lie within these outer bounds.
	Pass CGRectNull to not constrain the operation to outerbounds.
	
	\param targetAspectRatio
	if not 0.0, the CGRect will be constrained to this aspectRatio
 */
extern CGRect RMRectDragFromCorner(CGRect const theRect, CGPoint const endPoint, RMRectCorner const cornerIndex, CGRect const outerBounds, CGFloat const targetAspectRatio);

typedef RM_ENUM(NSUInteger, RMAutoresizingMask) {
	RMAutoresizingMaskNotSizable	= 0,
	RMAutoresizingMaskMinXMargin	= 1U << 0,
	RMAutoresizingMaskWidthSizable	= 1U << 1,
	RMAutoresizingMaskMaxXMargin	= 1U << 2,
	RMAutoresizingMaskMinYMargin	= 1U << 3,
	RMAutoresizingMaskHeightSizable	= 1U << 4,
	RMAutoresizingMaskMaxYMargin	= 1U << 5,
};
/*!
	\brief
	Applies the same distribution of dimensions between width/height and min/max x or y based on a resize between two sizes and an autoresizing mask
 */
extern CGRect RMRectAutoresizeRect(CGSize oldSize, CGSize newSize, CGRect rect, RMAutoresizingMask autoresizingMask);
